<?php

namespace Acelle\Http\Controllers\Api;

use Illuminate\Http\Request;
use Acelle\Http\Controllers\Controller;

/**
 * /api/v1/lists - API controller for managing lists.
 */
class MailListController extends Controller
{
    /**
     * Display all user's lists.
     *
     * GET /api/v1/lists
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $user = \Auth::guard('api')->user();

        // Get page and per_page from request, with default values
        $page = $request->input('page', 1);
        $perPage = $request->input('per_page', 10);

        // Get lists with pagination
        $listsQuery = \Acelle\Model\MailList::getAll()
            ->select('id', 'uid', 'name', 'from_email', 'from_name', 'status', 'created_at', 'updated_at')
            ->where('customer_id', '=', $user->customer->id);

        $total = $listsQuery->count();
        $lists = $listsQuery->skip(($page - 1) * $perPage)->take($perPage)->get();

        $lists = $lists->map(function ($list, $key) {
            return [
                'id' => $list->id,
                'uid' => $list->uid,
                'name' => $list->name,
                'from_email' => $list->from_email,
                'from_name' => $list->from_name,
                'subscribers' => $list->readCache('SubscriberCount', 0),
                'open_rate' => $list->readCache('UniqOpenRate', 0),
                'click_rate' => $list->readCache('ClickedRate', 0),
                'created_at' => $list->created_at,
                'updated_at' => $list->updated_at,
            ];
        });

        return \Response::json($lists, 200);
    }

    /**
     * Display the specified list information.
     *
     * GET /api/v1/lists/{id}
     *
     * @param int $id List's id
     *
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = \Auth::guard('api')->user();

        $item = \Acelle\Model\MailList::where('uid', '=', $id)
            ->first();

        // check if item exists
        if (!$item) {
            return \Response::json(array('message' => 'Mail list not found'), 404);
        }

        // authorize
        if (!$user->can('read', $item)) {
            return \Response::json(array('message' => 'Unauthorized'), 401);
        }

        // list info
        $list = [
            'uid' => $item->uid,
            'name' => $item->name,
            'from_email' => $item->from_email,
            'from_name' => $item->from_name,
            'remind_message' => $item->remind_message,
            'status' => $item->status,
            'created_at' => $item->created_at,
            'updated_at' => $item->updated_at,
        ];

        // List fields
        $list['fields'] = [];
        foreach ($item->getFields as $key => $field) {
            $list['fields'][] = [
                'key' => $field->tag, // for Zapier
                'label' => $field->label,
                'type' => 'string', // for Zapier
                'tag' => $field->tag,
                'default_value' => $field->default_value,
                'visible' => $field->visible,
                'required' => $field->required ? true : false,
            ];
        }

        // statistics
        $statistics = [
            'subscriber_count' => $item->subscribersCount(),
            'open_uniq_rate' => $item->openUniqRate(),
            'click_rate' => $item->clickRate(),
            'subscribe_rate' => $item->subscribeRate(),
            'unsubscribe_rate' => $item->unsubscribeRate(),
            'unsubscribe_count' => $item->unsubscribeCount(),
            'unconfirmed_count' => $item->unconfirmedCount(),
        ];

        return \Response::json(['list' => $list, 'statistics' => $statistics], 200);
    }

    /**
     * Create new list.
     *
     * POST /api/v1/lists/store
     *
     * @param \Illuminate\Http\Request $request All list information.
     *
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $user = \Auth::guard('api')->user();
        $list = new \Acelle\Model\MailList();

        // authorize
        if (!$user->can('create', $list)) {
            return \Response::json(array('status' => 0, 'message' => trans('no_more_item')), 403);
        }

        // validate and save posted data
        if ($request->isMethod('post')) {
            $validator = \Validator::make($request->all(), \Acelle\Model\MailList::$rules);
            if ($validator->fails()) {
                return response()->json($validator->messages(), 403);
            }

            // Save
            $list->fill($request->all());
            $list->customer_id = $user->customer->id;
            $list->save();

            // Log
            $list->log('created', $user->customer);

            // Trigger updating related campaigns cache
            $list->updateCachedInfo();

            return \Response::json(array(
                'status' => 1,
                'message' => trans('messages.list.created'),
                'list_uid' => $list->uid
            ), 200);
        }
    }

    /**
     * Add custom field for mail list.
     *
     * POST /api/v1/lists/store
     *
     * @param \Illuminate\Http\Request $request All list information.
     *
     * @return \Illuminate\Http\Response
     */
    public function addField(Request $request, $uid)
    {
        $user = \Auth::guard('api')->user();
        $list = \Acelle\Model\MailList::findByUid($uid);

        if (!$list) {
            return \Response::json(array('status' => 0, 'message' => 'Can not find list with uid=' . $uid), 404);
        }

        // authorize
        if (!$user->can('update', $list)) {
            return \Response::json(array('status' => 0, 'message' => trans('no_more_item')), 403);
        }

        // validate and save posted data
        if ($request->isMethod('post')) {
            $rules = [
                // check required input
                'type' => 'required|in:text,number,datetime',
                'label' => 'required',
                'tag' => 'required|alpha_dash',
            ];
            $validator = \Validator::make($request->all(), $rules);
            if ($validator->fails()) {
                return response()->json($validator->messages(), 403);
            }

            // try to add new field
            try {
                $field = $list->createField(
                    $type = $request->type,
                    $tag = $request->tag,
                    $label = $request->label,
                    $default_value = $request->default_value,
                    $requred = false,
                    $visible = true,
                );
            } catch (\Throwable $e) {
                return \Response::json(array('status' => 0, 'message' => $e->getMessage()), 403);
            }

            return \Response::json(array(
                'status' => 1,
                'message' => trans('messages.field.created'),
                'field' => [
                    'uid' => $field->uid,
                    'type' => $field->type,
                    'tag' => $field->tag,
                    'label' => $field->label,
                    'default_value' => $field->default_value,
                    'requred' => $field->requred,
                    'visible' => $field->visible,
                ],
            ), 200);
        }
    }

    public function delete($uid)
    {
        $user = \Auth::guard('api')->user();
        $list = \Acelle\Model\MailList::findByUid($uid);

        // check if item exists
        if (!$list) {
            return \Response::json(array('status' => 0, 'message' => 'Mail list not found'), 404);
        }

        // authorize
        if (!$user->can('delete', $list)) {
            return \Response::json(array('status' => 0, 'message' => 'Unauthorized'), 401);
        }

        $list->delete();

        // not needed as the related campaigns will be deleted as well
        // $item->updateCachedInfo();

        // Log
        $list->log('deleted', $user->customer);

        // update MailList cache
        event(new \Acelle\Events\MailListUpdated($list));

        return \Response::json(array('status' => 1, 'message' => 'Deleted'), 200);
    }
}
